/* LibTomCrypt, modular cryptographic library -- Tom St Denis
 *
 * LibTomCrypt is a library that provides various cryptographic
 * algorithms in a highly modular and flexible manner.
 *
 * The library is free for all purposes without any express
 * guarantee it works.
 *
 * Tom St Denis, tomstdenis@gmail.com, http://libtom.org
 */

/* Implements ECC over Z/pZ for curve y^2 = x^3 + ax + b
 *
 * All curves taken from NIST recommendation paper of July 1999
 * Available at http://csrc.nist.gov/cryptval/dss.htm
 */
#include "tomcrypt.h"

/**
  @file ecc.c
  ECC Crypto, Tom St Denis
*/  
/** Structure defines a NIST GF(p) curve */
   /**  1. domain id  - the domain id of the elliptic curve */
   /**  2. size       - The size of the curve in octets */
   /**  3. name       - name of curve */
   /**  4. prime      - The prime that defines the field the curve is in (encoded in base-64) */
   /**  5. A          - The x coefficient a (base64) */
   /**  6. B          - The fields B param (base64) */
   /**  7. order      - The order of the curve (base64) */
   /**  8. Gx         - The x co-ordinate of the base point on the curve (base64) */
   /**  9. Gy         - The y co-ordinate of the base point on the curve (base64) */

#ifdef LTC_MECC

/* This holds the key settings.  ***MUST*** be organized by size from smallest to largest. */
const ltc_ecc_set_type ltc_ecc_sets[] = {
#ifdef ECC112
{
        DomainID_secp112k1, // ITOS: Added during merge with LTC 1.17
        14,
        "SECP112R1",
        "DB7C2ABF62E35E668076BEAD208B",
		    "0000000000000000000000000000", // ITOS: Added during merge with LTC 1.17
        "659EF8BA043916EEDE8911702B22",
        "DB7C2ABF62E35E7628DFAC6561C5",
        "09487239995A5EE76B55F9C2F098",
        "A89CE5AF8724C0A23E0E0FF77500"
},
#endif
#ifdef ECC128
{
        DomainID_secp128k1, // ITOS: Added during merge with LTC 1.17
        16,
        "SECP128R1",
        "FFFFFFFDFFFFFFFFFFFFFFFFFFFFFFFF",
		    "00000000000000000000000000000000", // ITOS: Added during merge with LTC 1.17
        "E87579C11079F43DD824993C2CEE5ED3",
        "FFFFFFFE0000000075A30D1B9038A115",
        "161FF7528B899B2D0C28607CA52C5B86",
        "CF5AC8395BAFEB13C02DA292DDED7A83",
},
#endif
{
     DomainID_WMDRM10,
     20,
     "ECC-WMDRM10",
     "89ABCDEF012345672718281831415926141424F7",   /*prime*/
     "37A5ABCCD277BCE87632FF3D4780C009EBE41497",   /*A*/
     "DD8DABF725E2F3228E85F1AD78FDEDF9328239E",   /*B*/
     "89ABCDEF012345672716B26EEC14904428C2A675", /*order*/
     "8723947FD6A3A1E53510C07DBA38DAF0109FA120",   /*Gx*/
     "445744911075522D8C3C5856D4ED7ACDA379936F",   /*Gy*/
},
{
      DomainID_secp160k1,
      20,
     "ECC-160k1",
     "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFAC73",
     "0000000000000000000000000000000000000000",
     "0000000000000000000000000000000000000007",
     "0100000000000000000001B8FA16DFAB9ACA16B6B3",
     "3B4C382CE37AA192A4019E763036F4F5DD4D7EBB",
     "938CF935318FDCED6BC28286531733C3F03C4FEE",
},
{
     DomainID_secp160r1,
     20,
     "ECC-160r1",
     "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF7FFFFFFF",
     "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF7FFFFFFC",
     "1C97BEFC54BD7A8B65ACF89F81D4D4ADC565FA45",
     "0100000000000000000001F4C8F927AED3CA752257",
     "4A96B5688EF573284664698968C38BB913CBFC82",
     "23A628553168947D59DCC912042351377AC5FB32",
},
{
     DomainID_secp160r2,
     20,
     "ECC-160r2",
     "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFAC73",   /*prime*/
     "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFAC70",   /*A*/
     "B4E134D3FB59EB8BAB57274904664D5AF50388BA",   /*B*/
     "0100000000000000000000351EE786A818F3A1A16B", /*order*/
     "52DCB034293A117E1F4FF11B30F7199D3144CE6D",   /*Gx*/
     "FEAFFEF2E331F296E071FA0DF9982CFEA7D43F2E",   /*Gy*/
},
#ifdef ECC192K1
{
      DomainID_secp192k1,
      24,
      "ECC-192k1",
      "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFEE37",
      "000000000000000000000000000000000000000000000000",
      "000000000000000000000000000000000000000000000003",
      "FFFFFFFFFFFFFFFFFFFFFFFE26F2FC170F69466A74DEFD8D",
      "DB4FF10EC057E9AE26B07D0280B7F4341DA5D1B1EAE06C7D",
      "9B2F2F6D9C5628A7844163D015BE86344082AA88D95E2F9D",
},
#endif

#ifdef ECC192
{
        DomainID_secp192r1,
        24,
        "ECC-192r1",
        "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFF",
        "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFC",
        "64210519E59C80E70FA7E9AB72243049FEB8DEECC146B9B1",
        "FFFFFFFFFFFFFFFFFFFFFFFF99DEF836146BC9B1B4D22831",
        "188DA80EB03090F67CBF20EB43A18800F4FF0AFD82FF1012",
        "7192B95FFC8DA78631011ED6B24CDD573F977A11E794811",
},
#endif

{
        DomainID_secp224k1,
        28,
        "ECC-224k1",
        "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFE56D",
        "00000000000000000000000000000000000000000000000000000000",
        "00000000000000000000000000000000000000000000000000000005",
        "010000000000000000000000000001DCE8D2EC6184CAF0A971769FB1F7",
        "A1455B334DF099DF30FC28A169A467E9E47075A90F7E650EB6B7A45C",
        "7E089FED7FBA344282CAFBD6F7E319F7C0B0BD59E2CA4BDB556D61A5",
},

#ifdef ECC224
{
        DomainID_secp224r1,
        28,
        "ECC-224r1",
        "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF000000000000000000000001",
        "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFE",
        "B4050A850C04B3ABF54132565044B0B7D7BFD8BA270B39432355FFB4",
        "FFFFFFFFFFFFFFFFFFFFFFFFFFFF16A2E0B8F03E13DD29455C5C2A3D",
        "B70E0CBD6BB4BF7F321390B94A03C1D356C21122343280D6115C1D21",
        "BD376388B5F723FB4C22DFE6CD4375A05A07476444D5819985007E34",
},
#endif
{
        DomainID_secp256k1,
        32,
        "ECC-256k1",
        "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFC2F",
        "0000000000000000000000000000000000000000000000000000000000000000",
        "0000000000000000000000000000000000000000000000000000000000000007",
        "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEBAAEDCE6AF48A03BBFD25E8CD0364141",
        "79BE667EF9DCBBAC55A06295CE870B07029BFCDB2DCE28D959F2815B16F81798",
        "483ADA7726A3C4655DA4FBFC0E1108A8FD17B448A68554199C47D08FFB10D4B8",
},
#ifdef ECC256
{
        DomainID_secp256r1,
        32,
        "ECC-256r1",
        "FFFFFFFF00000001000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFF",
        "FFFFFFFF00000001000000000000000000000000FFFFFFFFFFFFFFFFFFFFFFFC",
        "5AC635D8AA3A93E7B3EBBD55769886BC651D06B0CC53B0F63BCE3C3E27D2604B",
        "FFFFFFFF00000000FFFFFFFFFFFFFFFFBCE6FAADA7179E84F3B9CAC2FC632551",
        "6B17D1F2E12C4247F8BCE6E563A440F277037D812DEB33A0F4A13945D898C296",
        "4FE342E2FE1A7F9B8EE7EB4A7C0F9E162BCE33576B315ECECBB6406837BF51F5",
},
#endif
#ifdef ECC384
{
        DomainID_secp384r1,
        48,
        "ECC-384r1",
        "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFF0000000000000000FFFFFFFF",
        "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFEFFFFFFFF0000000000000000FFFFFFFC",
        "B3312FA7E23EE7E4988E056BE3F82D19181D9C6EFE8141120314088F5013875AC656398D8A2ED19D2A85C8EDD3EC2AEF",
        "FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFC7634D81F4372DDF581A0DB248B0A77AECEC196ACCC52973",
        "AA87CA22BE8B05378EB1C71EF320AD746E1D3B628BA79B9859F741E082542A385502F25DBF55296C3A545E3872760AB7",
        "3617DE4A96262C6F5D9E98BF9292DC29F8F41DBD289A147CE9DA3113B5F0B8C00A60B1CE1D7E819D7A431D7C90EA0E5F",
},
#endif
#ifdef ECC521
{
        DomainID_secp521r1,
        66, // ITOS: There was 65 in LTC 1.13
        "ECC-521",
        "1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF",
        "01FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFC",
        "51953EB9618E1C9A1F929A21A0B68540EEA2DA725B99B315F3B8B489918EF109E156193951EC7E937B1652C0BD3BB1BF073573DF883D2C34F1EF451FD46B503F00",
        "1FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFA51868783BF2F966B7FCC0148F709A5D03BB5C9B8899C47AEBB6FB71E91386409",
        "C6858E06B70404E9CD9E3ECB662395B4429C648139053FB521F828AF606B4D3DBAA14B5E77EFE75928FE1DC127A2FFA8DE3348B3C1856A429BF97E7E31C2E5BD66",
        "11839296A789A3BC0045C8A5FB42C7D1BD998F54449579B446817AFBD17273E662C97EE72995EF42640C550B9013FAD0761353C7086A272C24088BE94769FD16650",
},
#endif
{
   0,0,
   NULL, NULL, NULL, NULL, NULL, NULL
}
};

#endif

/* $Source: /cvs/libtom/libtomcrypt/src/pk/ecc/ecc.c,v $ */
/* $Revision: 1.40 $ */
/* $Date: 2007/05/12 14:32:35 $ */

